<?php
/**
 * Abstract class for SEO Compatibility
 *
 * @package WCVendors_Pro
 *
 * @since 1.8.7
 * @version 1.8.7
 */
abstract class WCV_SEO_Compatibility {

    /**
     * WC Vendors SEO fields
     *
     * @since 1.8.7
     * @version 1.8.7
     *
     * @var array $store_seo_settings Vendors SEO settings array.
     */
    protected $store_seo_settings = array(
        'title'       => 'wcv_seo_title',
        'description' => 'wcv_seo_meta_description',
        'keywords'    => 'wcv_seo_meta_keywords',
        'fb_title'    => 'wcv_seo_fb_title',
        'fb_desc'     => 'wcv_seo_fb_description',
        'fb_image'    => 'wcv_seo_fb_image_id',
        'tw_title'    => 'wcv_seo_twitter_title',
        'tw_desc'     => 'wcv_seo_twitter_description',
        'tw_image'    => 'wcv_seo_twitter_image_id',
        'tw_site'     => '_wcv_twitter_username',
        'shop_name'   => 'pv_shop_name',
    );

    /**
     * Product SEO settings
     *
     * @since 1.8.7
     * @version 1.8.7
     * @var array $product_seo_settings Product SEO settings array.
     */
    protected $product_seo_settings = array(
        'title'       => 'wcv_product_seo_title',
        'description' => 'wcv_product_seo_description',
        'keywords'    => 'wcv_product_seo_keywords',
        'og'          => 'wcv_product_seo_opengraph',
        'tw_card'     => 'wcv_product_seo_twitter_card',
    );

    /**
     * Vendor id
     *
     * @since 1.8.7
     * @version 1.8.7
     *
     * @var int|bool $vendor_id Vendor id.
     */
    protected $vendor_id = false;

    /**
     * Is vendor store page
     *
     * @since 1.8.7
     * @version 1.8.7
     *
     * @var bool $is_vendor_store_page Is vendor store page.
     */
    protected $is_vendor_store_page = false;

    /**
     * Is product page
     *
     * @since 1.8.7
     * @version 1.8.7
     *
     * @var bool $is_product_page Is product page.
     */
    protected $is_product_page = false;

    /**
     * WC Vendors Pro SEO Compatibility constructor.
     *
     * @since 1.8.7
     * @version 1.8.7
     */
    public function __construct() {
        if ( ! $this->is_enable() ) {
            return;
        }
        $this->set_vendor_id();
        $this->is_vendor_store_page = WCV_Vendors::is_vendor_page();
        $this->is_product_page      = is_singular( 'product' ) && WCV_Vendors::is_vendor_product( wcv_get_user_role( $this->get_vendor_id() ) );
    }

    /**
     * Set vendor id
     *
     * @since 1.8.7
     * @version 1.8.7
     */
    public function set_vendor_id() {
        if ( WCV_Vendors::is_vendor_page() ) {
            $vendor_shop     = urldecode( get_query_var( 'vendor_shop' ) );
            $this->vendor_id = WCV_Vendors::get_vendor_id( $vendor_shop );
        }

        if ( is_singular( 'product' ) ) {
            $this->vendor_id = get_post_field( 'post_author', get_the_ID() );
        }
    }

    /**
     * Get vendor id
     *
     * @since 1.8.7
     * @version 1.8.7
     *
     * @return int|bool
     */
    public function get_vendor_id() {
        return $this->vendor_id;
    }

    /**
     * Abstract Is enable
     *
     * @since 1.8.7
     * @version 1.8.7
     */
    abstract public function is_enable();

    /**
     * Override site_name title
     *
     * @since 1.8.7
     * @version 1.8.7
     *
     * @param string $title The site name.
     * @return string $title The site name.
     */
    public function override_site_name_title( $title = '' ) {
        $store_title = get_user_meta( $this->get_vendor_id(), $this->store_seo_settings['shop_name'], true );
        if ( ! empty( $store_title ) ) {
            $title = $store_title;
        }
        return $title;
    }

    /**
     * Override twitter_site
     *
     * @since 1.8.7
     * @version 1.8.7
     *
     * @param string $site The twitter site.
     * @return string $site The twitter site.
     */
    public function override_twitter_site( $site = '' ) {
        $site_name = get_user_meta( $this->get_vendor_id(), $this->store_seo_settings['tw_site'], true );
        if ( ! empty( $site_name ) ) {
            $site = $site_name;
        }
        return $site;
    }

    /**
     * Override site title
     *
     * @since 1.8.7
     * @version 1.8.7
     *
     * @param string $title The site title.
     * @return string $title The site title.
     */
    public function override_site_title( $title = '' ) {
        $store_title = get_user_meta( $this->get_vendor_id(), $this->store_seo_settings['title'], true );
        if ( ! empty( $store_title ) ) {
            $title = $store_title;
        }
        return $title;
    }

    /**
     * Override og_title
     *
     * @since 1.8.7
     * @version 1.8.7
     *
     * @param string $title The og title.
     */
    public function override_store_og_title( $title = '' ) {
        $fb_title = get_user_meta( $this->get_vendor_id(), $this->store_seo_settings['fb_title'], true );
        if ( ! empty( $fb_title ) ) {
            $title = $fb_title;
        }
        return $title;
    }

    /**
     * Override og_description
     *
     * @since 1.8.7
     * @version 1.8.7
     *
     * @param string $description The og description.
     * @return string $description The og description.
     */
    public function override_store_og_description( $description = '' ) {
        $store_fb_desc = get_user_meta( $this->get_vendor_id(), $this->store_seo_settings['fb_desc'], true );
        if ( ! empty( $store_fb_desc ) ) {
            $description = $store_fb_desc;
        }
        return $description;
    }

    /**
     * Override og url
     *
     * @since 1.8.7
     * @version 1.8.7
     *
     * @param string $url The og url.
     */
    public function override_store_og_url( $url = '' ) {
        $store_url = WCV_Vendors::get_vendor_shop_page( $this->get_vendor_id() );
        if ( ! empty( $store_url ) ) {
            $url = $store_url;
        }
        return $url;
    }

    /**
     * Override og_image
     *
     * @since 1.8.7
     * @version 1.8.7
     *
     * @param string $image The og image.
     */
    public function override_store_og_image( $image = '' ) {
        $store_og_img_id = get_user_meta( $this->get_vendor_id(), $this->store_seo_settings['fb_image'], true );
        if ( ! empty( $store_og_img_id ) ) {
            $image = wp_get_attachment_url( $store_og_img_id );
        }
        return $image;
    }

    /**
     * Override twitter_title
     *
     * @since 1.8.7
     * @version 1.8.7
     *
     * @param string $title The twitter title.
     * @return string $title The twitter title.
     */
    public function override_store_twitter_title( $title = '' ) {
        $tw_title = get_user_meta( $this->get_vendor_id(), $this->store_seo_settings['tw_title'], true );
        if ( ! empty( $tw_title ) ) {
            $title = $tw_title;
        }
        return $title;
    }

    /**
     * Override twitter_description
     *
     * @since 1.8.7
     * @version 1.8.7
     *
     * @param string $description The twitter description.
     * @return string $description The twitter description.
     */
    public function override_store_twitter_description( $description = '' ) {
        $store_tw_desc = get_user_meta( $this->get_vendor_id(), $this->store_seo_settings['tw_desc'], true );
        if ( ! empty( $store_tw_desc ) ) {
            $description = $store_tw_desc;
        }
        return $description;
    }

    /**
     * Override twitter_image
     *
     * @since 1.8.7
     * @version 1.8.7
     *
     * @param string $image The twitter image.
     * @return string $image The twitter image.
     */
    public function override_store_twitter_image( $image = '' ) {
        $store_tw_img_id = get_user_meta( $this->get_vendor_id(), $this->store_seo_settings['tw_image'], true );
        if ( ! empty( $store_tw_img_id ) ) {
            $image = wp_get_attachment_url( $store_tw_img_id );
        }
        return $image;
    }

    /**
     * Override store description
     *
     * @since 1.8.7
     * @version 1.8.7
     *
     * @param string $description The store description.
     * @return string $description The store description.
     */
    public function override_store_description( $description = '' ) {
        $store_desc = get_user_meta( $this->get_vendor_id(), $this->store_seo_settings['description'], true );
        if ( ! empty( $store_desc ) ) {
            $description = $store_desc;
        }
        return $description;
    }

    /**
     *
     *
     * Product SEO SECTION
     */

    /**
     * Allow to output OG tags
     *
     * @since 1.8.7
     * @version 1.8.7
     *
     * @return bool True if OG tags are enabled, false otherwise.
     */
    public function is_og_enable() {
        return wc_string_to_bool( get_post_meta( get_the_ID(), $this->product_seo_settings['og'], 'no' ) );
    }

    /**
     * Allow to output twitter card
     *
     * @since 1.8.7
     * @version 1.8.7
     *
     * @return bool True if twitter card is enabled, false otherwise.
     */
    public function is_twitter_card_enable() {
        return wc_string_to_bool( get_post_meta( get_the_ID(), $this->product_seo_settings['tw_card'], 'no' ) );
    }


    /**
     * Override product og_title
     *
     * @since 1.8.7
     * @version 1.8.7
     *
     * @param string $title The og title.
     * @return string $title The og title.
     */
    public function override_product_og_title( $title = '' ) {

        $product_fb_title = get_post_meta( get_the_ID(), $this->product_seo_settings['title'], true );

        if ( ! empty( $product_fb_title ) ) {
            $title = $product_fb_title;
        }
        return $title;
    }

    /**
     * Override product og_description
     *
     * @since 1.8.7
     * @version 1.8.7
     *
     * @param string $description The og description.
     * @return string $description The og description.
     */
    public function override_product_og_description( $description = '' ) {
        $product_fb_desc = get_post_meta( get_the_ID(), $this->product_seo_settings['description'], true );
        if ( ! empty( $product_fb_desc ) ) {
            $description = $product_fb_desc;
        }
        return $description;
    }

    /**
     * Override product og_site_name
     *
     * @since 1.8.7
     * @version 1.8.7
     *
     * @param string $title The og site name.
     * @return string $title The og site name.
     */
    public function override_product_og_site_name( $title = '' ) {
        $site_name = get_user_meta( $this->get_vendor_id(), $this->store_seo_settings['shop_name'], true );
        if ( ! empty( $site_name ) ) {
                $title = $site_name;
        }
        return $title;
    }

    /**
     * Override product org_url
     *
     * @since 1.8.7
     * @version 1.8.7
     *
     * @param string $url The og url.
     * @return string $url The og url.
     */
    public function override_product_og_url( $url = '' ) {
        $product_url = WCV_Vendors::get_vendor_shop_page( $this->get_vendor_id() );
        if ( ! empty( $product_url ) ) {
            $url = $product_url;
        }
        return $url;
    }

    /**
     * Override product tw_title
     *
     * @since 1.8.7
     * @version 1.8.7
     *
     * @param string $title The twitter title.
     * @return string $title The twitter title.
     */
    public function override_product_twitter_title( $title = '' ) {
        $product_tw_title = $this->override_product_og_title( $title );
        if ( ! empty( $product_tw_title ) ) {
            $title = $product_tw_title;
        }
        return $title;
    }

    /**
     * Override product tw_description
     *
     * @since 1.8.7
     * @version 1.8.7
     *
     * @param string $description The twitter description.
     * @return string $description The twitter description.
     */
    public function override_product_twitter_description( $description = '' ) {
        $product_tw_desc = $this->override_product_og_description( $description );
        if ( ! empty( $product_tw_desc ) ) {
            $description = $product_tw_desc;
        }
        return $description;
    }

}
